/* NOTE: THE FOLLOWING example of anonymous function(s) CREATES A MASTER-MASTER REPLICATION SYSTEM.
          THE FOLLOWING FEATURES/CONCEPTS ARE INCLUDED IN THIS SNIPPET OF CODE:
            accessing postgres catalogs (pg_class, pg_namespace)
            composing & executing dynamic code (EXECUTE)
            creating views
            enforcing security (SECURITY DEFINER)
            replicating data between servers (dblink)
            rewrite rules
            grant rights and privileges (GRANT, REVOKE)
*/

-- upon ERROR condition condition, stop the script
\set ON_ERROR_STOP on

-- connect to the database as the superuser
\c mydatabase postgres

-- execute the following SQL as database owner
SET ROLE mydatabase;

-- remove prexisting schema
DROP SCHEMA IF EXISTS mydatabase_w CASCADE;

-- this anonymous function creates a view for every table in schema PUBLIC
DO
$$
DECLARE
  MYTABLE name;
  SQL     text;
BEGIN
  SQL:='create schema mydatabase_w' || E'\n';

  for MYTABLE in select relname
                  from pg_class a
                  join pg_namespace b on a.relnamespace=b.oid
                  where b.nspname like 'public'
                  and a.relkind like 'r'
                  order by relname
  loop
    SQL:=SQL || 'create view ' || MYTABLE || ' as select * from public.' || MYTABLE || E'\n';
  end loop;

  EXECUTE SQL;
END;
$$
LANGUAGE plpgsql;


-- create function call that makes remote connection
CREATE OR REPLACE FUNCTION mydatabase_w.get_conn (
  OUT conn text
) AS
$$
DECLARE
  PORT integer;
  HOST text;
BEGIN
  select setting into PORT from pg_settings where name like 'port';

  if    (PORT=10001) then
    PORT=11001;
    HOST='127.0.0.1';
  elsif (PORT=11001) then
    PORT=10001;
    HOST='127.0.0.1';
  end if;

  conn := 'hostaddr='|| HOST || ' port=' || PORT || ' dbname=mydatabase user=push connect_timeout=20 keepalives=0';
END;
$$
LANGUAGE plpgsql;

-- this function will be used as part of a rewrite rule which is attached to the VIEW
CREATE OR REPLACE FUNCTION mydatabase_w.rule_dml ()
    RETURNS VOID AS
$$
DECLARE
  SQL  text := current_query();         -- function call "current_query" traps the DML operation @ the VIEW
  CONN text := mydatabase_w.get_conn(); -- use the previously created connection
BEGIN
  EXECUTE SQL;
  PERFORM dblink_exec(CONN,SQL,true);   -- make the remote connection and execute the query SQL
  RETURN;
END;
$$
LANGUAGE plpgsql
SET SEARCH_PATH=PUBLIC
SECURITY DEFINER;                       -- function owned by postgres but can be executed by others


-- this anonymous function explicitly attaches the function mydatabase_w.rule_dml via
--      a rewrite rule that executes the SQL query  against both the table in schema PUBLIC
--      and the table @ the remote database
DO
$$
DECLARE
  MYTABLE name;
BEGIN
  for MYTABLE in select relname
                  from pg_class a
                  join pg_namespace b on a.relnamespace=b.oid
                  where b.nspname like 'mydatabase_w'
                  and a.relkind like 'v'
                  order by relname
  loop
    EXECUTE 'create rule dml_insert as on insert to mydatabase_w.'||MYTABLE||' do instead select mydatabase_w.rule_dml()';
    EXECUTE 'create rule dml_update as on update to mydatabase_w.'||MYTABLE||' do instead select mydatabase_w.rule_dml()';
    EXECUTE 'create rule dml_delete as on delete to mydatabase_w.'||MYTABLE||' do instead select mydatabase_w.rule_dml()';
  end loop;
END;
$$
LANGUAGE plpgsql;

-- reset session role back to postgres and set privileges
RESET ROLE;
GRANT USAGE ON SCHEMA mydatabase_w TO mydatabase_w;
GRANT ALL PRIVILEGES ON ALL TABLES IN SCHEMA mydatabase_w TO mydatabase_w;
ALTER FUNCTION mydatabase_w.rule_dml() OWNER TO postgres;

